/** @file   car.h
 * @brief   Declaration of Car - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_CAR_H
#define H_WWW_CAR_H

#include "polygonobject.h"

namespace WeWantWar {


/** @class  Car
 * @brief   The cars in city levels are instances of this class.
 * @author  Tomi Lamminsaari
 *
 */
class Car : public PolygonObject
{
public:

  ///
  /// Constants and static members
  /// ============================

  /** Possible types of the car. Allows us to draw different cars.
   */
  enum CarType {
    CAR_TYPE1,
    CAR_TYPE2,
    CAR_TYPE3,
    CAR_TYPE4,
    CAR_TYPE5,
    CAR_TYPE6,
    CAR_TYPE_POLICE,
    CAR_TYPE7,
    CAR_TYPE8
  };
  
  
  
  ///
  /// Constructor, destructor
  /// =======================
  
  /** class constructor
   * @param     t                 Type of the car
   */
  Car( CarType t );
  
  /** Constructs new car and defines the route it should drive
   * @param     t                 Type of the car
   * @param     routename         Name of the route.
   */
  Car( CarType t, const std::string& routename );
  
  /** class destructor
   */
  virtual ~Car();
  
  
  
  ///
  /// Public methods
  /// ==============
  
  /** Updates the car's operations.
   */
  virtual void update();
  

  /** Kills the car.
   */
  virtual void kill();
  
  /** Makes sound
   * @param     id                ID of the sound we should make
   */
  virtual void makeSound( GameObject::SoundID id ) const;
  
  /** Checks if the given bullet hits us.
   * @param     pB                Pointer to Bullet
   * @return    <code>true</code> if bullet did hit us.
   */
  virtual bool hitByBullet( Bullet* pB );
  
  /** Sets the route for this car. This works only, if our controller
   * supports it.
   * @param     routename         Name of the route
   */
  void setRoute( const std::string& routename );
  

  
  ///
  /// Getter methods
  /// ==============
  
  /** Tells if this object is currently reloading.
   * @return    Always 'false' since cars don't shoot
   */
  virtual bool reloading() const;
  
  /** Returns the type of this object.
   * @return    ObjectID::TYPE_CAR
   */
  virtual ObjectID::Type objectType() const;
  
protected:
  ///
  /// Protected methods
  /// =================
  
  /** Initilizes the parameters of this car. Creates the polygons based on
   * the value in <code>m_carType</code> - member. Sets the armor and
   * other data.
   * @return    Nonzero if fails.
   */
  int prepareCar();
  
  ///
  /// Members
  /// =======

  /** The type of this car-object. */
  CarType     m_carType;
  
  /** The driving velocity of this car. */
  eng2d::Vec2D  m_movementVec;
  
  
private:

  Car( const Car& rC );
  Car& operator = ( const Car& rC );
};

};  // end of namespace

#endif // CAR_H

/**
 * Version history
 * ===============
 * $Log: car.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:16:50+02  lamminsa
 * Initial revision
 *
 */
 
